// Reaction-Diffusion Paint Applet by Golan Levin
// (c) 1998, MIT Media Laboratory
// This program allows the user to paint into an RD simulation.
// The simulation was ported from the 1991 C code by Greg Turk. 

import java.applet.*;
import java.awt.*;
import java.util.*;
import java.net.*;
import java.io.*;
import java.awt.image.*;

public class Chemicals extends Applet implements Runnable
{
	Thread animThread = null;
	boolean dieThread;
	private int frameRate = 30;
	private Image canvasImage;
	
	private static boolean okay = true;
	private static final int mandatorySleepTime = 6;
	private static final int availableTimePerFrame = 20; // milliseconds; sums to 30 => 33fps
	private	static final String creditParam = "credit";
	private static String creditValue;

	// parameters for the canvas image
	private static final int canvasImageLeft = 0;
	private static final int canvasImageTop = 0;
	private static final int canvasImageWidth = 144;
	private static final int canvasImageHeight = canvasImageWidth; //64; // must be square
	private static final int canvasImageRight = canvasImageLeft + canvasImageWidth;
	private static final int canvasImageBottom = canvasImageTop + canvasImageHeight;
	private static final int xrepeats = 4;
	private static final int yrepeats = 3;
	private static final int appWidth = xrepeats*canvasImageWidth;
	private static final int appHeight = yrepeats*canvasImageHeight;

	// mouse-related variables
	private static boolean theMouseDown = false;
	private static int prevx, prevy; 
	private static int newx, newy;

	// establishing squares
	private static int squareSize;
	private static int xSquares, ySquares;

	// parameters for the imaging
	private static IndexColorModel myColorModel;
	private static MemoryImageSource mis;
	private static byte pixels[];
	private static int	sourcePixels[];
	private static int  nPixels;
	
	
	/* simulation variables */
	private static int xsize = canvasImageWidth;
	private static int ysize = canvasImageHeight;
	private static int interval = 1;
	private static int value_switch = 9;
	private static int MAX = canvasImageWidth; //64; //must be equal to canvasImageWidth & canvasImageHeight
	private static float seedSize = 5.0f;
	
	private static float[][] a = new float [MAX][MAX];
	private static float[][] b = new float [MAX][MAX];
	private static float[][] c = new float [MAX][MAX];
	private static float[][] d = new float [MAX][MAX];
	private static float[][] e = new float [MAX][MAX];

	private static float[][] da = new float [MAX][MAX];
	private static float[][] db = new float [MAX][MAX];
	private static float[][] dc = new float [MAX][MAX];
	private static float[][] dd = new float [MAX][MAX];
	private static float[][] de = new float [MAX][MAX];

	private static float[][] ai = new float [MAX][MAX];

	private static float p1,p2,p3;
	private static float diff1,  diff2;
	private static float diff14, diff24;

	private static float arand;
	private static float speed = 1.0f;
	private static int k=0;


	//////////////////////////////////////////////////////////////////
	public Chemicals(){}

	public void start(){
		dieThread = false;
		requestFocus();
		if (animThread == null){ 
			animThread = new Thread (this);
			animThread.start();
		} 
	}
	
	public void stop(){ 
		update();
		dieThread = true;
		if (animThread != null){
			animThread.stop();
			animThread = null;
		}
	}
	
	public void destroy(){;}
	public void update(){ paint(this.getGraphics()); }
	public void update(Graphics g){ paint(g); }


	public void init() {
		authenticate(); 
		resize(appWidth, appHeight);
		canvasImage = createImage (canvasImageWidth, canvasImageHeight);
		
		// create color palette
		float val = 0;
		byte[] r, g, b;
		r= new byte[256]; g= new byte[256]; b= new byte[256];
		for  (int i=0; i<256; i++) {
			val = (float)i/256f;
			r[i] = (byte) Math.round (Math.pow(val, 0.3) * 255.0f); //1.0 output
			g[i] = (byte) Math.round (val * 204.0f); //0.8 output
			b[i] = (byte) Math.round (Math.pow(val, 0.6) * 153.0f); //0.6 output, 
		} 
		myColorModel = new IndexColorModel(8,  256,  r,  g,  b);
		
		// deal with images
		nPixels = canvasImageWidth*canvasImageHeight;
		pixels = new byte[nPixels];
		for (int i=0; i<nPixels; i++){ pixels[i] = 0;}
		mis = new MemoryImageSource(canvasImageWidth, canvasImageHeight, myColorModel, pixels, 0, canvasImageWidth);
	    mis.setAnimated(true);	    
		canvasImage = createImage(mis);
		
		// init simulation
		do_stripes();
		

		prevx =0;
		prevy =0; 
		newx = 0;
		newy = 0;
	}
	
	
	
	
	private void authenticate(){
		okay = true;
		
		/*
		boolean accreditation = false;
		boolean authentication = false;
		
		// make sure the accreditation is correct in the html page parameter
		creditValue	= getParameter(creditParam);
		if ((creditValue.hashCode() == (68625034*2)) &&
			(creditValue.length() == (30+9))){ accreditation = true; }
		else { 
			accreditation = false;
			creditValue = "no good. credit parameter has been modified.";
			System.out.println(creditValue);
		}
		okay = accreditation;
		*/
		
		/*
		// make sure the special goodie can be found, and checks out
		final char c0 = 'm'; 
		final char c1 = 'h'; 
		final char c2 = 't';
		final char c3 = 'l'; 
		final char c4 = '.'; // lmth.html
		final String goodie = ((getCodeBase()).toString() + c3+c0+c2+c1+c4+c1+c2+c0+c3);
		try {
			DataInputStream ds = null;
			URL fileURL = new URL(goodie);
			ds = new DataInputStream(fileURL.openStream());
			String line = ds.readLine();
			authentication = true;
			if ((line.hashCode() ==  ((326299559*2)+1)) &&
				(line.length() == (2*7))) {
				authentication = true;
			} else { 
				authentication = false;
				creditValue = "no good. applet has been transplanted.";
				System.out.println(creditValue);
			}
			ds.close();
		}
		catch (IOException e){
			creditValue = "no good. applet transplanted.";
			System.out.println(creditValue);
			authentication = false;
		}

		if ((accreditation==true) && 
			(authentication==true)) { okay = true; }
		*/
	}


	
	public synchronized void paint(Graphics g) {
		if (okay){
			try { 
					for (int i=0; i<yrepeats; i++){
						for (int j=0; j<xrepeats; j++){
							int yloc = i*canvasImageHeight;
							int xloc = j*canvasImageWidth;
							g.drawImage( canvasImage, xloc, yloc, this); 
						}
					}
		
				} 
			catch (NullPointerException e) { 
				System.out.println("error:  "  +  e.getMessage());
			}
		} else { // applet not OK
			try { 
				g.setColor(Color.black);
				g.fillRect(0, 0, appWidth, appHeight);
			}
			catch (NullPointerException e) { 
				System.out.println("error:  "  +  e.getMessage());
			}
			this.getAppletContext().showStatus(creditValue);
		}
	}
	

	
	private synchronized void updatePixelsBogus(){
		int value;
		int loc=0;
		int r=0; 
		int g=0;
		int b=0;
		
		double scale = 255.0/Math.PI;
		for (int y=0; y<canvasImageBottom; y++){
			for (int x=0; x<canvasImageRight; x++){
				loc = y*canvasImageWidth +x;
				value = 0xff000000;
				
				r = (int) (Math.cos(x/scale)*127 + 127);
				g = (int) (Math.sin(y/scale)*127 + 127);
				b = (int) (Math.sin(k/5.0)*127 + 127);
				//value |= (r << 16) | (g << 8) | b;
				value = (r+b+g)/3;
				pixels[loc] = (byte) value;
			}
		}
		mis.newPixels(0, 0, canvasImageWidth, canvasImageHeight);
					
	}
	
	
	
	

	////////////////////////////////////////////////////////////////////////////////////
	
	// Init params for Meinhardt's stripe-formation system.
	private void do_stripes(){
		p1 = 0.04f;
		p2 = 0.06f;
		p3 = 0.04f;

		diff1 = 0.009f;
		diff2 = 0.2f;
		diff14 = 4f * diff1;
		diff24 = 4f * diff2;

		arand = 0.02f;
		
		// calculate semistable equilibria
		semi_equilibria();
	}
	
	
	
	
	// Diffuse and react.
	private synchronized void updatePixels(){  // was 'compute'
		
		// this.getAppletContext().showStatus("iteration " + k);
		// start things diffusing
		//if (k % interval == 0) {
			switch (value_switch) {
				case 1:	show(a);	break;
				case 2: show(b);	break;
				case 3: show(c);	break;
				case 4: show(d);	break;
				case 5: show(e);	break;
					
				case 6:	show(da);	break;
				case 7: show(db);	break;
				case 8: show(dc);	break;
				case 9: show(dd);	break;
				case 10: show(de);	break;
						
				case 11: show(ai);	break;
			}
		//}

		// perform reaction and diffusion
		multiplicative_help();
		k++;
	}



	// Create stripes with what Hans Meinhardt calls a two-species balance.
	private synchronized void multiplicative_help() {
		int	i, j;
		int	iprev, inext, jprev, jnext;
		float aval, bval, cval, dval, eval;
		float ka, kc, kd;
		float temp1, temp2;
		float dda, ddb;
		float ddd, dde;
		
		int ysizem1 = ysize - 1;
		int xsizem1 = xsize - 1;
		
		// compute change in each cell

		for (i = 0; i < xsize; i++) {

			ka = -p1 - diff14;
			kc = -p2;
			kd = -p3 - diff24;

			iprev = (i==0)? xsizem1 : i-1;
			inext = (i==xsizem1) ? 0 : i+1;
			//-----------------------------------------
			// j == 0 case
			j = 0;
			jprev = ysize - 1;
			jnext = 1;

			aval = a[i][j];
			bval = b[i][j];
			cval = c[i][j];
			dval = d[i][j];
			eval = e[i][j];

			temp1 = 0.01f * aval * aval * eval * ai[i][j];//0.01
			temp2 = 0.01f * bval * bval * dval;

			dda = a[i][jprev] + a[i][jnext] + a[iprev][j] + a[inext][j];
			ddb = b[i][jprev] + b[i][jnext] + b[iprev][j] + b[inext][j];
			ddd = d[i][jprev] + d[i][jnext] + d[iprev][j] + d[inext][j];
			dde = e[i][jprev] + e[i][jnext] + e[iprev][j] + e[inext][j];

			da[i][j] = aval * ka + diff1 * dda + temp1 / cval;
			db[i][j] = bval * ka + diff1 * ddb + temp2 / cval;
			dc[i][j] = cval * kc + temp1 + temp2;
			dd[i][j] = dval * kd + diff2 * ddd + p3 * aval;
			de[i][j] = eval * kd + diff2 * dde + p3 * bval;
			
			
			//-----------------------------------------
			// j regular case
			jprev = 0;
			for (j = 1; j < ysizem1; j++) {
				jnext = j+1;

				aval = a[i][j];
				bval = b[i][j];
				cval = c[i][j];
				dval = d[i][j];
				eval = e[i][j];

				temp1 = 0.01f * aval * aval * eval * ai[i][j];//0.01
				temp2 = 0.01f * bval * bval * dval;

				dda = a[i][jprev] + a[i][jnext] + a[iprev][j] + a[inext][j];
				ddb = b[i][jprev] + b[i][jnext] + b[iprev][j] + b[inext][j];
				ddd = d[i][jprev] + d[i][jnext] + d[iprev][j] + d[inext][j];
				dde = e[i][jprev] + e[i][jnext] + e[iprev][j] + e[inext][j];

				da[i][j] = aval * ka + diff1 * dda + temp1 / cval;
				db[i][j] = bval * ka + diff1 * ddb + temp2 / cval;
				dc[i][j] = cval * kc + temp1 + temp2;
				dd[i][j] = dval * kd + diff2 * ddd + p3 * aval;
				de[i][j] = eval * kd + diff2 * dde + p3 * bval;
				
				jprev = j;
			}
			
			
			
			//-----------------------------------------
			// j == ysizem1 case
			j = ysizem1;
			jprev = ysizem1 -1;
			jnext = 0;

			aval = a[i][j];
			bval = b[i][j];
			cval = c[i][j];
			dval = d[i][j];
			eval = e[i][j];

			temp1 = 0.01f * aval * aval * eval * ai[i][j];//0.01
			temp2 = 0.01f * bval * bval * dval;

			dda = a[i][jprev] + a[i][jnext] + a[iprev][j] + a[inext][j];
			ddb = b[i][jprev] + b[i][jnext] + b[iprev][j] + b[inext][j];
			ddd = d[i][jprev] + d[i][jnext] + d[iprev][j] + d[inext][j];
			dde = e[i][jprev] + e[i][jnext] + e[iprev][j] + e[inext][j];

			da[i][j] = aval * ka + diff1 * dda + temp1 / cval;
			db[i][j] = bval * ka + diff1 * ddb + temp2 / cval;
			dc[i][j] = cval * kc + temp1 + temp2;
			dd[i][j] = dval * kd + diff2 * ddd + p3 * aval;
			de[i][j] = eval * kd + diff2 * dde + p3 * bval;
			
			
			
			
			
		}

		// affect change
		for (i = 0; i < xsize; i++){
			for (j = 0; j < ysize; j++) {
				a[i][j] += (speed * da[i][j]);
				b[i][j] += (speed * db[i][j]);
				c[i][j] += (speed * dc[i][j]);
				d[i][j] += (speed * dd[i][j]);					
				e[i][j] += (speed * de[i][j]);
			}
		}
	}
	
	
	//Calculate semi-stable equilibria.
	private void semi_equilibria(){
		int i,j;
		float ainit, binit, cinit, dinit, einit;
		ainit = binit = cinit = dinit = einit = 0;
		
		ainit = p2 / (2.0f * p1);
		binit = ainit;
		cinit = 0.02f * ainit * ainit * ainit / p2;
		dinit = ainit;
		einit = ainit;
		
		// figure the values
		for (i = 0; i < xsize; i++) {
			for (j = 0; j < ysize; j++) {
				a[i][j] = ainit;
				b[i][j] = binit;
				c[i][j] = cinit;
				d[i][j] = dinit;
				e[i][j] = einit;
				ai[i][j] = 1.0f;
				//ai[i][j] = 1.0f + frand (-0.5f * arand, 0.5f * arand);
			}
		}
	}
	
	
	
	// generate a random number between min and max
	private float frand(float min, float max){
		return (float) (min + Math.random() * (max - min));
	}
		
	
	
	
	
	// Display the activator.
	private synchronized void show(float[][] values){
		// values has dimensions [MAX][MAX]!
		int i,j;
		float output;
		float min =  Float.MAX_VALUE;
		float max =  Float.MIN_VALUE;
		// find minimum and maximum values
		for (i = 0; i < xsize; i++){
			for (j = 0; j < ysize; j++) {
				if (values[i][j] < min) { min = values[i][j];}
				else if (values[i][j] > max) { max = values[i][j];}
			}
		}

		if (min == max) {
			min = max - 1;
			max = min + 2;
		}

		// display the values! 
		int loc = 0;
		float dif =	255.0f / (max - min);
		for (i = 0; i < xsize; i++){
			for (j = 0; j < ysize; j++) {
				pixels[loc++] = (byte) ((values[i][j] - min) * dif);	
			}
		}
		mis.newPixels(0, 0, canvasImageWidth, canvasImageHeight);
	}
	

	
	public void run(){ 
		while (animThread != null) {
			long then = System.currentTimeMillis();
			if ((theMouseDown == true) && 
				(prevx - newx == 0) && (prevy - newy == 0)){
				int i = newx%canvasImageWidth;
				int j = newy%canvasImageHeight;
				ai[j][i] += seedSize;
			}
			updatePixels();
			update();
			
			try {
				if (!dieThread) { // includes dynamic sleep by Tom White
					 try {Thread.sleep(mandatorySleepTime);} 
					 // sleep for a minimum of mandatorySleepTime millisecs per frame
					 catch(InterruptedException e){}
					 long now = System.currentTimeMillis(); 
					 int napTime = (int) (availableTimePerFrame - (now - then)); 
					 // sleep or work for at least availableTimePerFrame more
					 if (napTime > 0) {
						 try { Thread.sleep(napTime);}
						 catch (InterruptedException e) {}
					 }
				} else {
					if (animThread != null) {
						animThread.stop();
						animThread = null;
						a = null;
						b = null;
						c = null;
						d = null;
						e = null;
						da = null;
						db = null;
						dc = null;
						dd = null;
						de = null;
						break;
					}
				}
			} catch (Exception e) { break;}
		}
	}








	///////////////////////////////////////////////////
	public boolean action(Event evt, Object arg) {
		return false; // action wasn't processed here
	}

	public synchronized boolean mouseDown (Event evt, int x, int y) {
		/*
		for (int i=0; i<MAX; i++){
			for (int j=0; j<MAX; j++){
				ai[i][j] = 1.0f;
			}
		}
		*/
		theMouseDown = true;
		prevx = x;
		prevy = y; 
		newx = x;
		newy = y;
		int i = x%canvasImageWidth;
		int j = y%canvasImageHeight;
		ai[j][i] += seedSize;
		return true;
	}

	public boolean mouseDrag (Event evt, int x, int y) {
		prevx = newx;
		prevy = newy;
		newx = x;
		newy = y;
		
		
		bresenhamDrawLine (prevx, prevy, newx, newy);
		//int i = x%canvasImageWidth;
		//int j = y%canvasImageHeight;
		//ai[j][i] += seedSize;
		return true;
	}
	
	void bresenhamDrawLine (int xP, int yP, int xQ, int yQ){
		// draw a line by placing pixels, by Bresenham's algorithm.
		// from "Computer Graphics for Java Programmers" by Ammeraal
		
		int x=xP, y=yP, D=0;
		int HX=xQ-xP, HY=yQ-yP;
		int xInc=1, yInc=1;
		int c, M;
		
		float vel = (float) (0.5 * Math.log (HX*HX + HY*HY));
		
		if (HX < 0){xInc = -1; HX = -HX;}
		if (HY < 0){yInc = -1; HY = -HY;}
		if (HY <= HX){
			c = 2*HX; M = 2*HY;
			for (;;) { // I hate this syntax
				if (x==xQ) break; // slight modification so that the final point isn't actually drawn
				ai[y%MAX][x%MAX] += (seedSize + vel); // put the pixel
				x+= xInc;
				D+= M;
				if (D > HX){y += yInc; D-=c;}
			}
		}
		else {
			c = 2*HY; M = 2*HX;
			for (;;) { // grr
				if (y==yQ) break; 
				// slight modification so that the final point isn't actually drawn/
				// usually the test would follow the putPixel, but I switch their order here
				// because otherwise the endpoint would be drawn twice 
				// (when the next line segment is made)
				ai[y%MAX][x%MAX] += (seedSize + vel); // put the pixel
				y+= yInc;
				D+= M;
				if (D > HY){x += xInc; D-=c;}
			}
		}
	}

	

	public boolean mouseMove (Event evt, int x, int y) {
		theMouseDown = false;
		return true;
	}

	public boolean mouseUp (Event evt, int x, int y) { 
		theMouseDown = false;
		newx = x;
		newy = y;
		return true;
	}

	public synchronized boolean keyDown(Event evt, int code) {
		if ((code == 13) || (code == 10)) { // return,newline,enter
			if (value_switch == 9){ value_switch = 1;}
			else if (value_switch == 1){ value_switch = 9;}
			else {value_switch = 9;}

		} else {
			switch ((char)code){
			case ' ': 
				semi_equilibria();
				break;
			case 'a': value_switch = 11; break;
		
			/*
			case '1': value_switch = 1; break;
			case '2': value_switch = 2; break;
			case '3': value_switch = 3; break;
			case '4': value_switch = 4; break;
			case '5': value_switch = 5; break;
			case '6': value_switch = 6; break;
			case '7': value_switch = 7; break;
			case '8': value_switch = 8; break;
			case '9': value_switch = 9; break;
			case '0': value_switch = 10; break;
			
			 */
		
			}
		}
		return true;
	}



}


///////////////////////////////////////////////////////////////////
/*

//	some ideas:
//	-- damping on velocities
//	-- method for clearing the screen

//	-- try adding to other arrays, not just ai[][], as in:
//	-- a diffusion map: user draws into a pixmap which modulates
		the diffusion rate of the chemicals at a given pixel
//	-- seeds (current method): fluctuations from equilibrium are placed where
		the user draws; these become kernels around which the simulation expands
//	-- direct adding to chemical saturation at a given pixel, and the FREEZING of the
		values of those pixels.

*/
